package structr

import (
	"fmt"
	"testing"

	"codeberg.org/gruf/go-byteutil"
	"codeberg.org/gruf/go-kv/format"
)

func TestQueue(t *testing.T) {
	t.Run("structA", func(t *testing.T) { testQueue(t, testStructA) })
	t.Run("structB", func(t *testing.T) { testQueue(t, testStructB) })
	t.Run("structC", func(t *testing.T) { testQueue(t, testStructC) })
}

func testQueue[T any](t *testing.T, test test[T]) {
	var q Queue[*T]

	// Create invalidate function hook
	// to track invalidated value ptrs.
	popped := make(map[string]bool)
	popFn := func(value *T) {
		var buf byteutil.Buffer
		format.Appendf(&buf, "{:?}", value)
		popped[buf.String()] = true
	}
	wasPopped := func(value *T) bool {
		var buf byteutil.Buffer
		format.Appendf(&buf, "{:?}", value)
		return popped[buf.String()]
	}

	// Initialize the struct queue.
	q.Init(QueueConfig[*T]{
		Indices: test.indices,
		Pop:     popFn,
	})

	// Check that fake indices cause panic
	for _, index := range test.indices {
		fake := index.Fields + "!"
		catchpanic(t, func() {
			q.Index(fake)
		}, "unknown index: "+fake)
	}

	// Check that wrong
	// index causes panic
	catchpanic(t, func() {
		wrong := new(Index)
		q.Pop(wrong)
	}, "invalid index for queue")

	// Push all values to front.
	t.Logf("PushFront: %v", test.values)
	q.PushFront(test.values...)

	// Ensure queue length of expected size.
	if l := q.Len(); l != len(test.values) {
		t.Fatalf("queue not of expected length: have=%d want=%d", l, len(test.values))
	}

	// Check values in expected order.
	for _, value := range test.values {
		check, ok := q.PopBack()
		t.Logf("PopBack: %+v", check)
		if !ok || !test.equalfn(value, check) {
			t.Fatalf("value not at expected location: value=%+v check=%+v", value, check)
		}
	}

	// Check that queue is empty.
	if l := q.Len(); l != 0 {
		t.Fatalf("queue should be empty: was=%d", l)
	}

	// Ensure all values were popped
	// on pop via the callback function.
	for _, value := range test.values {
		if !wasPopped(value) {
			t.Fatalf("expected value was not popped: %+v", value)
		}
	}

	// Reset popped.
	clear(popped)

	// Push all values to back.
	t.Logf("PushBack: %v", test.values)
	q.PushBack(test.values...)

	// Ensure queue length of expected size.
	if l := q.Len(); l != len(test.values) {
		t.Fatalf("queue not of expected length: have=%d want=%d", l, len(test.values))
	}

	// Check values in expected order.
	for _, value := range test.values {
		check, ok := q.PopFront()
		t.Logf("PopFront: %+v", check)
		if !ok || !test.equalfn(value, check) {
			t.Fatalf("value not at expected location: value=%+v check=%+v", value, check)
		}
	}

	// Check that queue is empty.
	if l := q.Len(); l != 0 {
		t.Fatalf("queue should be empty: was=%d", l)
	}

	// Ensure all values were popped
	// on pop via the callback function.
	for _, value := range test.values {
		if !wasPopped(value) {
			t.Fatalf("expected value was not popped: %+v", value)
		}
	}

	// Reset popped.
	clear(popped)

	// Push all values to back.
	t.Logf("PushFront: %v", test.values)
	q.PushFront(test.values...)

	// Pop each of the values from the queue
	// by their indexed key. It's easier to just
	// iterate through all the values for all indices
	// instead of getting particular about which
	// value is stored in which particular index.
	for _, index := range test.indices {
		var keys []Key

		// Get associated structr index.
		idx := q.Index(index.Fields)

		for _, value := range test.values {
			// extract key parts for value.
			parts, ok := indexkey(idx, value)
			if !ok {
				continue
			}

			// generate key from parts.
			key := idx.Key(parts...)

			// add index key to keys.
			keys = append(keys, key)
		}

		// Pop all keys in index.
		t.Logf("Pop: %s %v", index.Fields, keys)
		_ = q.Pop(idx, keys...)
	}

	// Ensure all values were popped
	// on pop via the callback function.
	for _, value := range test.values {
		if !wasPopped(value) {
			t.Fatalf("expected value was not popped: %+v", value)
		}
	}

	// Reset popped.
	clear(popped)

	// print final debug.
	fmt.Println(q.Debug())
}
